/*
 * Decompiled with CFR 0.152.
 */
package org.ehcache.shadow.org.terracotta.offheapstore.paging;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.Lock;
import org.ehcache.shadow.org.terracotta.offheapstore.paging.Page;
import org.ehcache.shadow.org.terracotta.offheapstore.paging.PageSource;
import org.ehcache.shadow.org.terracotta.offheapstore.storage.PointerSize;
import org.ehcache.shadow.org.terracotta.offheapstore.storage.allocator.Allocator;
import org.ehcache.shadow.org.terracotta.offheapstore.storage.allocator.IntegerBestFitAllocator;
import org.ehcache.shadow.org.terracotta.offheapstore.storage.allocator.LongBestFitAllocator;
import org.ehcache.shadow.org.terracotta.offheapstore.util.DebuggingUtils;
import org.ehcache.shadow.org.terracotta.offheapstore.util.Validation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OffHeapStorageArea {
    private static final Logger LOGGER = LoggerFactory.getLogger(OffHeapStorageArea.class);
    private static final boolean VALIDATING = Validation.shouldValidate(OffHeapStorageArea.class);
    private static final long LARGEST_POWER_OF_TWO = Integer.highestOneBit(Integer.MAX_VALUE);
    private static final ByteBuffer[] EMPTY_BUFFER_ARRAY = new ByteBuffer[0];
    private final int initialPageSize;
    private final int maximalPageSize;
    private final int pageGrowthAreaSize;
    private final float compressThreshold;
    private final Owner owner;
    private final PageSource pageSource;
    private final Allocator allocator;
    private final Random random = new Random();
    private Deque<Collection<Page>> released = new LinkedList<Collection<Page>>();
    private final Map<Integer, Page> pages = new ConcurrentHashMap<Integer, Page>(1, 0.75f, 1);
    private final boolean thief;
    private final boolean victim;

    public OffHeapStorageArea(PointerSize width, Owner owner, PageSource pageSource, int pageSize, boolean thief, boolean victim) {
        this(width, owner, pageSource, pageSize, pageSize, thief, victim);
    }

    public OffHeapStorageArea(PointerSize width, Owner owner, PageSource pageSource, int pageSize, boolean thief, boolean victim, float compressThreshold) {
        this(width, owner, pageSource, pageSize, pageSize, thief, victim, compressThreshold);
    }

    public OffHeapStorageArea(PointerSize width, Owner owner, PageSource pageSource, int initialPageSize, int maximalPageSize, boolean thief, boolean victim) {
        this(width, owner, pageSource, initialPageSize, maximalPageSize, thief, victim, 0.0f);
    }

    public OffHeapStorageArea(PointerSize width, Owner owner, PageSource pageSource, int initialPageSize, int maximalPageSize, boolean thief, boolean victim, float compressThreshold) {
        if (victim && maximalPageSize != initialPageSize) {
            throw new IllegalArgumentException("Variable page-size offheap storage areas cannot be victims as they do not support stealing.");
        }
        this.owner = owner;
        this.pageSource = pageSource;
        switch (width) {
            case INT: {
                this.allocator = new IntegerBestFitAllocator(this);
                break;
            }
            case LONG: {
                this.allocator = new LongBestFitAllocator(this);
                break;
            }
            default: {
                throw new UnsupportedOperationException();
            }
        }
        initialPageSize = Math.max(this.allocator.getMinimalSize(), initialPageSize);
        this.initialPageSize = Integer.bitCount(initialPageSize) == 1 ? (int)Math.min(LARGEST_POWER_OF_TWO, (long)initialPageSize) : (int)Math.min(LARGEST_POWER_OF_TWO, Long.highestOneBit(initialPageSize) << 1);
        this.maximalPageSize = maximalPageSize < initialPageSize ? initialPageSize : (Integer.bitCount(maximalPageSize) == 1 ? (int)Math.min(LARGEST_POWER_OF_TWO, (long)maximalPageSize) : (int)Math.min(LARGEST_POWER_OF_TWO, Long.highestOneBit(maximalPageSize) << 1));
        this.pageGrowthAreaSize = this.maximalPageSize - this.initialPageSize;
        this.compressThreshold = compressThreshold;
        this.thief = thief;
        this.victim = victim;
    }

    public void clear() {
        this.allocator.clear();
        Iterator<Page> it = this.pages.values().iterator();
        while (it.hasNext()) {
            Page p = it.next();
            it.remove();
            this.freePage(p);
        }
        this.validatePages();
    }

    public byte readByte(long address) {
        int pageIndex = this.pageIndexFor(address);
        int pageAddress = this.pageAddressFor(address);
        return this.pages.get(pageIndex).asByteBuffer().get(pageAddress);
    }

    public short readShort(long address) {
        int pageSize;
        int pageIndex = this.pageIndexFor(address);
        int pageAddress = this.pageAddressFor(address);
        if (pageAddress + 2 <= (pageSize = this.pageSizeFor(pageIndex))) {
            return this.pages.get(pageIndex).asByteBuffer().getShort(pageAddress);
        }
        short value = 0;
        for (int i = 0; i < 2; ++i) {
            ByteBuffer buffer = this.pages.get(pageIndex).asByteBuffer();
            value = (short)(value | (0xFF & buffer.get(pageAddress)) << 8 * (1 - i));
            pageIndex = this.pageIndexFor(++address);
            pageAddress = this.pageAddressFor(address);
        }
        return value;
    }

    public int readInt(long address) {
        int pageSize;
        int pageIndex = this.pageIndexFor(address);
        int pageAddress = this.pageAddressFor(address);
        if (pageAddress + 4 <= (pageSize = this.pageSizeFor(pageIndex))) {
            return this.pages.get(pageIndex).asByteBuffer().getInt(pageAddress);
        }
        int value = 0;
        for (int i = 0; i < 4; ++i) {
            ByteBuffer buffer = this.pages.get(pageIndex).asByteBuffer();
            value |= (0xFF & buffer.get(pageAddress)) << 8 * (3 - i);
            pageIndex = this.pageIndexFor(++address);
            pageAddress = this.pageAddressFor(address);
        }
        return value;
    }

    public long readLong(long address) {
        int pageSize;
        int pageIndex = this.pageIndexFor(address);
        int pageAddress = this.pageAddressFor(address);
        if (pageAddress + 8 <= (pageSize = this.pageSizeFor(pageIndex))) {
            return this.pages.get(pageIndex).asByteBuffer().getLong(pageAddress);
        }
        long value = 0L;
        for (int i = 0; i < 8; ++i) {
            ByteBuffer buffer = this.pages.get(pageIndex).asByteBuffer();
            value |= (0xFFL & (long)buffer.get(pageAddress)) << 8 * (7 - i);
            pageIndex = this.pageIndexFor(++address);
            pageAddress = this.pageAddressFor(address);
        }
        return value;
    }

    public ByteBuffer readBuffer(long address, int length) {
        ByteBuffer[] buffers = this.readBuffers(address, length);
        if (buffers.length == 1) {
            return buffers[0];
        }
        ByteBuffer copy = ByteBuffer.allocate(length);
        for (ByteBuffer b : buffers) {
            copy.put(b);
        }
        return ((ByteBuffer)copy.flip()).asReadOnlyBuffer();
    }

    public ByteBuffer[] readBuffers(long address, int length) {
        ByteBuffer buffer;
        int pageSize;
        int pageIndex = this.pageIndexFor(address);
        int pageAddress = this.pageAddressFor(address);
        if (pageAddress + length <= (pageSize = this.pageSizeFor(pageIndex))) {
            ByteBuffer pageBuffer = this.pages.get(pageIndex).asByteBuffer().duplicate();
            ByteBuffer buffer2 = ((ByteBuffer)pageBuffer.limit(pageAddress + length).position(pageAddress)).slice().asReadOnlyBuffer();
            return new ByteBuffer[]{buffer2};
        }
        ArrayList<ByteBuffer> buffers = new ArrayList<ByteBuffer>(length / pageSize);
        for (int remaining = length; remaining > 0; remaining -= buffer.remaining()) {
            ByteBuffer pageBuffer = this.pages.get(pageIndex).asByteBuffer().duplicate();
            pageBuffer.position(pageAddress);
            if (pageBuffer.remaining() > remaining) {
                pageBuffer.limit(pageBuffer.position() + remaining);
            }
            buffer = pageBuffer.slice().asReadOnlyBuffer();
            buffers.add(buffer);
            pageIndex = this.pageIndexFor(address += (long)buffer.remaining());
            pageAddress = this.pageAddressFor(address);
        }
        return buffers.toArray(EMPTY_BUFFER_ARRAY);
    }

    public void writeByte(long address, byte value) {
        int pageIndex = this.pageIndexFor(address);
        int pageAddress = this.pageAddressFor(address);
        this.pages.get(pageIndex).asByteBuffer().put(pageAddress, value);
    }

    public void writeShort(long address, short value) {
        int pageSize;
        int pageIndex = this.pageIndexFor(address);
        int pageAddress = this.pageAddressFor(address);
        if (pageAddress + 2 <= (pageSize = this.pageSizeFor(pageIndex))) {
            this.pages.get(pageIndex).asByteBuffer().putShort(pageAddress, value);
        } else {
            for (int i = 0; i < 2; ++i) {
                ByteBuffer buffer = this.pages.get(pageIndex).asByteBuffer();
                buffer.position(pageAddress);
                buffer.put((byte)(value >> 8 * (1 - i)));
                pageIndex = this.pageIndexFor(++address);
                pageAddress = this.pageAddressFor(address);
            }
        }
    }

    public void writeInt(long address, int value) {
        int pageSize;
        int pageIndex = this.pageIndexFor(address);
        int pageAddress = this.pageAddressFor(address);
        if (pageAddress + 4 <= (pageSize = this.pageSizeFor(pageIndex))) {
            this.pages.get(pageIndex).asByteBuffer().putInt(pageAddress, value);
        } else {
            for (int i = 0; i < 4; ++i) {
                ByteBuffer buffer = this.pages.get(pageIndex).asByteBuffer();
                buffer.position(pageAddress);
                buffer.put((byte)(value >> 8 * (3 - i)));
                pageIndex = this.pageIndexFor(++address);
                pageAddress = this.pageAddressFor(address);
            }
        }
    }

    public void writeLong(long address, long value) {
        int pageSize;
        int pageIndex = this.pageIndexFor(address);
        int pageAddress = this.pageAddressFor(address);
        if (pageAddress + 8 <= (pageSize = this.pageSizeFor(pageIndex))) {
            this.pages.get(pageIndex).asByteBuffer().putLong(pageAddress, value);
        } else {
            for (int i = 0; i < 8; ++i) {
                ByteBuffer buffer = this.pages.get(pageIndex).asByteBuffer();
                buffer.position(pageAddress);
                buffer.put((byte)(value >> 8 * (7 - i)));
                pageIndex = this.pageIndexFor(++address);
                pageAddress = this.pageAddressFor(address);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void writeBuffer(long address, ByteBuffer data) {
        int pageIndex = this.pageIndexFor(address);
        int pageAddress = this.pageAddressFor(address);
        int pageSize = this.pageSizeFor(pageIndex);
        if (pageAddress + data.remaining() <= pageSize) {
            ByteBuffer buffer = this.pages.get(pageIndex).asByteBuffer();
            buffer.position(pageAddress);
            buffer.put(data);
        } else {
            while (data.hasRemaining()) {
                ByteBuffer buffer = this.pages.get(pageIndex).asByteBuffer();
                buffer.position(pageAddress);
                if (data.remaining() > buffer.remaining()) {
                    int originalLimit = data.limit();
                    try {
                        data.limit(data.position() + buffer.remaining());
                        address += (long)data.remaining();
                        buffer.put(data);
                    }
                    finally {
                        data.limit(originalLimit);
                    }
                } else {
                    address += (long)data.remaining();
                    buffer.put(data);
                }
                pageIndex = this.pageIndexFor(address);
                pageAddress = this.pageAddressFor(address);
            }
        }
    }

    public void writeBuffers(long address, ByteBuffer[] data) {
        for (ByteBuffer buffer : data) {
            int length = buffer.remaining();
            this.writeBuffer(address, buffer);
            address += (long)length;
        }
    }

    public void free(long address) {
        float occupation;
        this.allocator.free(address);
        if (this.compressThreshold > 0.0f && (occupation = (float)this.getOccupiedMemory() / (float)this.allocator.getLastUsedAddress()) < this.compressThreshold) {
            this.compress();
        }
    }

    private boolean compress() {
        long lastAddress = this.allocator.getLastUsedPointer();
        int sizeOfArea = this.owner.sizeOf(lastAddress);
        long compressed = this.allocator.allocate(sizeOfArea);
        if (compressed >= 0L) {
            if (compressed < lastAddress) {
                this.writeBuffers(compressed, this.readBuffers(lastAddress, sizeOfArea));
                if (this.owner.moved(lastAddress, compressed)) {
                    this.allocator.free(lastAddress);
                    return true;
                }
            }
            this.allocator.free(compressed);
            return false;
        }
        return false;
    }

    public void destroy() {
        this.allocator.clear();
        Iterator<Page> it = this.pages.values().iterator();
        while (it.hasNext()) {
            Page p = it.next();
            it.remove();
            this.freePage(p);
        }
        this.validatePages();
    }

    public long allocate(long size) {
        do {
            long address;
            if ((address = this.allocator.allocate(size)) < 0L) continue;
            return address;
        } while (this.expandData());
        return -1L;
    }

    private boolean expandData() {
        int newPageSize = this.nextPageSize();
        if (this.getAllocatedMemory() + (long)newPageSize > this.allocator.getMaximumAddress()) {
            return false;
        }
        Page newPage = this.pageSource.allocate(newPageSize, this.thief, this.victim, this);
        if (newPage == null) {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Data area expansion from {} failed", (Object)this.getAllocatedMemory());
            }
            return false;
        }
        if (this.pages.put(this.pages.size(), newPage) == null) {
            this.validatePages();
            this.allocator.expand(newPageSize);
            if (LOGGER.isDebugEnabled()) {
                long before = this.getAllocatedMemory();
                long after = before + (long)newPageSize;
                LOGGER.debug("Data area expanded from {}B to {}B [occupation={}]", new Object[]{DebuggingUtils.toBase2SuffixedString(before), DebuggingUtils.toBase2SuffixedString(after), Float.valueOf((float)this.allocator.occupied() / (float)after)});
            }
            return true;
        }
        this.freePage(newPage);
        this.validatePages();
        throw new AssertionError();
    }

    public long getAllocatedMemory() {
        return this.addressForPage(this.pages.size());
    }

    public long getOccupiedMemory() {
        return this.allocator.occupied();
    }

    public String toString() {
        Page p;
        StringBuilder sb = new StringBuilder("OffHeapStorageArea\n");
        int i = 0;
        while (i < this.pages.size() && (p = this.pages.get(i++)) != null) {
            Page q;
            int size = p.size();
            int count = 1;
            while (i < this.pages.size() && (q = this.pages.get(i)) != null && q.size() == size) {
                ++count;
                ++i;
            }
            sb.append("\t").append(count).append(" ").append(DebuggingUtils.toBase2SuffixedString(size)).append("B page").append(count == 1 ? "\n" : "s\n");
        }
        sb.append("Allocator: ").append(this.allocator).append('\n');
        sb.append("Page Source: ").append(this.pageSource);
        return sb.toString();
    }

    private int pageIndexFor(long address) {
        if (address > (long)this.pageGrowthAreaSize) {
            return (int)((address - (long)this.pageGrowthAreaSize) / (long)this.maximalPageSize + (long)this.pageIndexFor(this.pageGrowthAreaSize));
        }
        return 64 - Long.numberOfLeadingZeros(address / (long)this.initialPageSize + 1L) - 1;
    }

    private long addressForPage(int index) {
        int postIndex = index - this.pageIndexFor(this.pageGrowthAreaSize);
        if (postIndex > 0) {
            return (long)this.pageGrowthAreaSize + (long)this.maximalPageSize * (long)postIndex;
        }
        return (this.initialPageSize << index) - this.initialPageSize;
    }

    private int pageAddressFor(long address) {
        return (int)(address - this.addressForPage(this.pageIndexFor(address)));
    }

    private int pageSizeFor(int index) {
        if (index < this.pageIndexFor(this.pageGrowthAreaSize)) {
            return this.initialPageSize << index;
        }
        return this.maximalPageSize;
    }

    private int nextPageSize() {
        return this.pageSizeFor(this.pages.size());
    }

    public void validateStorageArea() {
        this.allocator.validateAllocator();
    }

    public void release(long address) {
        int lastPage = this.pageIndexFor(address);
        for (int i = this.pages.size() - 1; i > lastPage; --i) {
            Page p = this.pages.remove(i);
            this.allocator.expand(-p.size());
            this.freePage(p);
        }
        this.validatePages();
    }

    private void freePage(Page p) {
        if (this.released.isEmpty()) {
            this.pageSource.free(p);
        } else {
            this.released.peek().add(p);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Collection<Page> release(Collection<Page> targets) {
        Lock ownerLock = this.owner.writeLock();
        if (this.thief || this.owner.isThief()) {
            if (!ownerLock.tryLock()) {
                return Collections.emptyList();
            }
        } else {
            ownerLock.lock();
        }
        try {
            LinkedList<Object> recovered = new LinkedList<Object>();
            LinkedList<Object> freed = new LinkedList<Object>();
            while (freed.size() < targets.size()) {
                long remove = this.allocator.getLastUsedPointer();
                if (remove < 0L) {
                    for (int i = this.pages.size() - 1; i >= 0; --i) {
                        Page free = this.pages.get(i);
                        this.allocator.expand(-free.size());
                        this.pages.remove(i);
                        if (targets.remove(free)) {
                            recovered.add(free);
                            continue;
                        }
                        freed.add(free);
                    }
                    this.validatePages();
                    break;
                }
                ArrayList releasedPages = new ArrayList();
                this.released.push(releasedPages);
                try {
                    if (this.owner.evictAtAddress(remove, true) || this.moveAddressDown(remove)) {
                        for (Page p : releasedPages) {
                            if (targets.remove(p)) {
                                recovered.add(p);
                                continue;
                            }
                            freed.add(p);
                        }
                        this.validatePages();
                        continue;
                    }
                    if (!releasedPages.isEmpty()) {
                        throw new AssertionError();
                    }
                    break;
                }
                finally {
                    this.released.pop();
                }
            }
            Iterator freePageSource = freed.iterator();
            for (Page t : targets) {
                int index = this.getIndexForPage(t);
                if (index < 0 || !freePageSource.hasNext()) continue;
                Page f = (Page)freePageSource.next();
                Validation.validate(!VALIDATING || f != t);
                Validation.validate(!VALIDATING || f.size() == t.size());
                ((ByteBuffer)f.asByteBuffer().clear()).put((ByteBuffer)t.asByteBuffer().clear());
                this.pages.put(index, f);
                recovered.add(t);
            }
            this.validatePages();
            while (freePageSource.hasNext()) {
                this.freePage((Page)freePageSource.next());
            }
            LinkedList<Object> linkedList = recovered;
            return linkedList;
        }
        finally {
            ownerLock.unlock();
        }
    }

    private boolean moveAddressDown(long target) {
        int sizeOfArea = this.owner.sizeOf(target);
        long ceiling = this.addressForPage(Math.max(0, this.pageIndexFor(target) - 2)) + 1L;
        long startAt = this.random.nextLong() % ceiling;
        Iterator pointers = this.allocator.iterator();
        while (pointers.hasNext() && (Long)pointers.next() < startAt) {
        }
        while (pointers.hasNext()) {
            long relocated;
            long address = (Long)pointers.next();
            if (address >= target || !this.owner.evictAtAddress(address, false) || (relocated = this.allocator.allocate(sizeOfArea)) < 0L) continue;
            if (relocated < target) {
                this.writeBuffers(relocated, this.readBuffers(target, sizeOfArea));
                if (!this.owner.moved(target, relocated)) {
                    throw new AssertionError((Object)"Failure to move mapping during release");
                }
                this.allocator.free(target);
                return true;
            }
            this.allocator.free(relocated);
        }
        LOGGER.debug("Random Eviction Failure Migration Failed - Using Biased Approach");
        Iterator iterator = this.allocator.iterator();
        while (iterator.hasNext()) {
            long relocated;
            long address = (Long)iterator.next();
            if (address >= target || !this.owner.evictAtAddress(address, false) || (relocated = this.allocator.allocate(sizeOfArea)) < 0L) continue;
            if (relocated < target) {
                this.writeBuffer(relocated, this.readBuffer(target, sizeOfArea));
                this.owner.moved(target, relocated);
                this.allocator.free(target);
                return true;
            }
            this.allocator.free(relocated);
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean shrink() {
        Lock ownerLock = this.owner.writeLock();
        ownerLock.lock();
        try {
            if (this.pages.isEmpty()) {
                boolean bl = false;
                return bl;
            }
            int initialSize = this.pages.size();
            for (Page p : this.release(new LinkedList<Page>(Collections.singletonList(this.pages.get(this.pages.size() - 1))))) {
                this.freePage(p);
            }
            boolean bl = this.pages.size() < initialSize;
            return bl;
        }
        finally {
            ownerLock.unlock();
        }
    }

    private int getIndexForPage(Page p) {
        for (Map.Entry<Integer, Page> e : this.pages.entrySet()) {
            if (e.getValue() != p) continue;
            return e.getKey();
        }
        return -1;
    }

    private void validatePages() {
        if (VALIDATING) {
            for (int i = 0; i < this.pages.size(); ++i) {
                if (this.pages.get(i) != null) continue;
                ArrayList<Integer> pageIndices = new ArrayList<Integer>(this.pages.keySet());
                Collections.sort(pageIndices);
                throw new AssertionError((Object)("Page Indices " + pageIndices));
            }
        }
    }

    public static interface Owner {
        public boolean evictAtAddress(long var1, boolean var3);

        public Lock writeLock();

        public boolean isThief();

        public boolean moved(long var1, long var3);

        public int sizeOf(long var1);
    }
}

